﻿#nullable enable
using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading.Tasks;
using Tessa.UI.Cards;
using Tessa.UI.Cards.Controls.AutoComplete;
using Tessa.Views;
using Tessa.Views.Metadata;
using Tessa.Views.Metadata.Criteria;

namespace Tessa.Extensions.AclExamples.Client.UI
{
    public class MeetingProtocolUIExtension(IViewService viewService) : CardUIExtension
    {
        #region Nested Types

        private sealed class RowInfo(
            List<object?> row,
            Dictionary<string, int> indicesByName)
        {
            public T Get<T>(string column)
            {
                int index = indicesByName[column];
                return (T) row[index]!;
            }
        }

        #endregion

        #region Constants

        private const string CollegiatePartControlAlias = "CollegiatePartControl";
        private const string AuthorControlAlias = "AuthorControl";

        private const string CollegiatePartsViewAlias = "AclExamplesCollegiateParts";
        private const string AclUsersViewAlias = "AclUsers";

        #endregion

        #region Fields

        private readonly IViewService viewService = NotNullOrThrow(viewService);

        private ITessaView? collegiatePartsView;

        private ITessaView? aclUsersView;

        #endregion

        #region Base Overrides

        public override Task Initialized(ICardUIExtensionContext context)
        {
            if (context.Model.Controls.TryGet<AutoCompleteEntryViewModel>(CollegiatePartControlAlias) is not { } collegiatePartControl
                || context.Model.Controls.TryGet<AutoCompleteEntryViewModel>(AuthorControlAlias) is not { } authorControl)
            {
                return Task.CompletedTask;
            }

            var aclMeetingProtocolsSection = context.Card.Sections["AclMeetingProtocols"];

            collegiatePartControl.ValueSelected += async (sender, args) =>
            {
                var newCollegiatePart = args.Item.Reference;
                if (this.collegiatePartsView is null)
                {
                    if ((this.collegiatePartsView =
                            await this.viewService
                                .GetByNameAsync(CollegiatePartsViewAlias, context.CancellationToken)
                                .ConfigureAwait(false)) is null)
                    {
                        // у пользователя нет доступа к представлению, или нет самого представления
                        return;
                    }
                }

                var collegiatePartsViewMetadata = await this.collegiatePartsView.GetMetadataAsync(context.CancellationToken).ConfigureAwait(false);

                // получаем данные представления
                var viewResult = await this.collegiatePartsView.GetDataAsync(
                    new TessaViewRequest(collegiatePartsViewMetadata.Alias)
                    {
                        // состояние заданий = "Новое"
                        new RequestParameter("CollegiatePartIDParam").Add(EqualsToCriteriaOperator.Instance, newCollegiatePart)
                    }, context.CancellationToken);

                if (viewResult.Rows.Count == 0)
                {
                    return;
                }

                var indicesByName = viewResult.CreateColumnIndicesByName();

                var row = new RowInfo(viewResult.Rows.First(), indicesByName);
                var organizationID = row.Get<Guid>("OrganizationID");
                var organizationName = row.Get<string>("OrganizationName");

                if ((Guid?) aclMeetingProtocolsSection.RawFields["CollegiatePartOrganizationID"] != organizationID)
                {
                    aclMeetingProtocolsSection.Fields["CollegiatePartOrganizationID"] = organizationID;
                    aclMeetingProtocolsSection.Fields["CollegiatePartOrganizationName"] = organizationName;
                }
            };

            authorControl.ValueSelected += async (sender, args) =>
            {
                var newAuthor = args.Item.Reference;

                if (this.aclUsersView is null)
                {
                    if ((this.aclUsersView =
                            await this.viewService
                                .GetByNameAsync(AclUsersViewAlias, context.CancellationToken)
                                .ConfigureAwait(false)) is null)
                    {
                        // у пользователя нет доступа к представлению, или нет самого представления
                        return;
                    }
                }

                IViewMetadata aclUsersViewViewMetadata = await this.aclUsersView.GetMetadataAsync(context.CancellationToken).ConfigureAwait(false);
                ITessaViewRequest aclUsersViewViewRequest = new TessaViewRequest(aclUsersViewViewMetadata.Alias);

                // состояние заданий = "Новое"
                aclUsersViewViewRequest.Parameters.Add(new RequestParameter("UserID").Add(EqualsToCriteriaOperator.Instance, newAuthor));

                // получаем данные представления
                var viewResult = await this.aclUsersView.GetDataAsync(aclUsersViewViewRequest, context.CancellationToken);
                if (viewResult.Rows.Count == 0)
                {
                    return;
                }

                var indicesByName = viewResult.CreateColumnIndicesByName();

                var row = new RowInfo(viewResult.Rows.First(), indicesByName);
                var departmentID = row.Get<Guid?>("DepartmentID");
                var departmentName = row.Get<string>("DepartmentName");
                var organizationID = row.Get<Guid?>("OrganizationID");
                var organizationName = row.Get<string>("OrganizationName");

                if ((Guid?) aclMeetingProtocolsSection.RawFields["AuthorDepartmentID"] != departmentID)
                {
                    aclMeetingProtocolsSection.Fields["AuthorDepartmentID"] = departmentID;
                    aclMeetingProtocolsSection.Fields["AuthorDepartmentName"] = departmentName;
                }

                if ((Guid?) aclMeetingProtocolsSection.RawFields["AuthorOrganizationID"] != organizationID)
                {
                    aclMeetingProtocolsSection.Fields["AuthorOrganizationID"] = organizationID;
                    aclMeetingProtocolsSection.Fields["AuthorOrganizationName"] = organizationName;
                }
            };

            return Task.CompletedTask;
        }

        #endregion
    }
}
